/*
* @name Drawing
* @description Generative painting program.
* Adapted from https://editor.p5js.org/p5/sketches/Hello_P5:_drawing
*/

// All the paths
let paths = [];
// Are we painting?
let painting = true;
// How long until the next circle
let nextMin = 250;
let nextMax = 1000;
// Where are we now and where were we?
let current;
let previous;

let lastTimeStamp = 0;

let canvas;
let parent;

function setup() {
    const viewPort = getViewPortSize();
    canvas = createCanvas(viewPort.x, viewPort.y);
    current = createVector(0,0);
    previous = createVector(0,0);

    paths.push(new Path());
}

function draw() {
    background(255);

    // If it's time for a new point
    let waitTime = random(nextMin, nextMax);
    if (millis() - lastTimeStamp > waitTime && painting) {
        lastTimeStamp = millis();

        // Grab mouse position
        current.x = random(canvas.width);
        current.y = random(canvas.height);

        // New particle's force is based on mouse movement
        let force = p5.Vector.sub(current, previous);
        force.mult(0.05);

        // Add new particle
        paths[paths.length - 1].add(current, force);

        // Schedule next circle
        next = millis() + random(100);

        // Store mouse values
        previous.x = current.x;
        previous.y = current.y;
    }

    // Draw all paths
    for( let i = 0; i < paths.length; i++) {
        paths[i].update();
        paths[i].display();
    }
}

function windowResized() {
    const viewPort = getViewPortSize();
    resizeCanvas(viewPort.x, viewPort.y);
}

function getViewPortSize() {
    const vw = Math.max(document.documentElement.clientWidth, window.innerWidth || 0);
    const vh = Math.max(document.documentElement.clientHeight, window.innerHeight || 0);
    return createVector(vw, vh);
}

// A Path is a list of particles
class Path {
    constructor() {
        this.particles = [];
        this.hue = random(100);
    }

    add(position, force) {
        // Add a new particle with a position, force, and hue
        this.particles.push(new Particle(position, force, this.hue));
    }

    // Display plath
    update() {
        for (let i = 0; i < this.particles.length; i++) {
            this.particles[i].update();
        }
    }

    // Display plath
    display() {
        // Loop through backwards
        for (let i = this.particles.length - 1; i >= 0; i--) {
            // If we shold remove it
            if (this.particles[i].lifespan <= 0) {
                this.particles.splice(i, 1);
                // Otherwise, display it
            } else {
                this.particles[i].display(this.particles[i+1]);
            }
        }

    }
}

// Particles along the path
class Particle {
    constructor(position, force, hue) {
        this.position = createVector(position.x, position.y);
        this.velocity = createVector(force.x, force.y);
        this.drag = 0.95;
        this.lifespan = 255;
    }

    update() {
        // Move it
        this.position.add(this.velocity);
        // Slow it down
        this.velocity.mult(this.drag);
        // Fade it out
        this.lifespan--;
    }

    // Draw particle and connect it with a line
    // Draw a line to another
    display(other) {
        stroke(0, this.lifespan);
        fill(0, this.lifespan/2);
        ellipse(this.position.x,this.position.y, 8, 8);
        // If we need to draw a line
        if (other) {
            line(this.position.x, this.position.y, other.position.x, other.position.y);
        }
    }
}
